# frozen_string_literal: true

require 'spec_helper'
require_relative '../lib/test_info'

RSpec.describe TestInfo do
  describe '.test_supported_by_gitlab_settings?' do
    subject(:result) do
      described_class.test_supported_by_gitlab_settings?(
        test_file,
        gitlab_env_settings,
        env_url,
        rate_limits,
        unattended
      )
    end

    let(:test_file) { '/path/to/some_test.js' }
    let(:env_url) { 'https://gitlab.example.com' }
    let(:rate_limits) { false }
    let(:unattended) { true }

    before do
      allow(described_class).to receive(:get_test_tag_value)
        .with(test_file, 'gitlab_settings')
        .and_return(test_required_settings)

      allow(described_class).to receive(:warn)
      allow(described_class).to receive(:disable_rate_limit_setting)
      allow(GPTCommon).to receive(:show_warning_prompt)
      allow(GPTCommon).to receive(:change_env_settings)
      allow(Rainbow).to receive(:new)
      allow(GPTLogger.logger).to receive(:error)
      allow(ENV).to receive(:[]).with('ACCESS_TOKEN').and_return('test_token')
    end

    context 'when test_required_settings is nil' do
      let(:test_required_settings) { nil }
      let(:gitlab_env_settings) { { 'some_setting' => 'value' } }

      it 'returns true (test is supported)' do
        expect(result).to be true
      end
    end

    describe 'with invalid values in test_required_settings' do
      shared_examples 'it handles invalid test_required_settings' do |description, value|
        context "when test_required_settings is #{description}" do
          let(:test_required_settings) { value }
          let(:gitlab_env_settings) { { 'some_setting' => 'value' } }

          it "returns false (test is not supported)" do
            expect(result).to be(false)
          end
        end
      end

      include_examples 'it handles invalid test_required_settings', 'not parseable JSON', "this is a string"
      include_examples 'it handles invalid test_required_settings', 'empty string', ''
      include_examples 'it handles invalid test_required_settings', 'empty object', {}
    end

    describe 'with empty gitlab_env_settings' do
      let(:test_required_settings) { '{"some_setting": "value"}' }

      shared_examples 'it handles empty gitlab environment settings' do |description, value|
        context "when gitlab_env_settings is #{description}" do
          let(:gitlab_env_settings) { value }

          it 'returns false (test is not supported)' do
            expect(result).to be false
          end
        end
      end

      include_examples 'it handles empty gitlab environment settings', 'nil', nil
      include_examples 'it handles empty gitlab environment settings', 'empty hash', {}
      include_examples 'it handles empty gitlab environment settings', 'empty string', ''
      include_examples 'it handles empty gitlab environment settings', 'not JSON', 'string value'
    end

    describe 'when a setting in test @gitlab_settings tag that matches the GitLab instance settings' do
      shared_examples 'it supports tests with matching settings' do |description, required_settings, actual_settings|
        context "with #{description}" do
          let(:test_required_settings) { required_settings }
          let(:gitlab_env_settings) { actual_settings }

          it 'returns true (test is supported)' do
            expect(result).to be true
          end
        end
      end

      include_examples 'it supports tests with matching settings',
        'boolean true',
        '{"boolean_value_positive": true}',
        { 'boolean_value_positive' => true }

      include_examples 'it supports tests with matching settings',
        'boolean false',
        '{"boolean_value_negative": false}',
        { 'boolean_value_negative' => false }

      include_examples 'it supports tests with matching settings',
        'numeric value',
        '{"numeric_value": 100}',
        { 'numeric_value' => 100 }

      include_examples 'it supports tests with matching settings',
        'string value',
        '{"string_value": "this-is-a-string"}',
        { 'string_value' => 'this-is-a-string' }

      include_examples 'it supports tests with matching settings',
        'complex nested object',
        '{"config": {"a": true, "b": 5}}',
        { 'config' => { 'a' => true, 'b' => 5 } }

      include_examples 'it supports tests with matching settings',
        'array',
        '{"allowed_values": ["a", "b", "c"]}',
        { 'allowed_values' => %w[a b c] }

      include_examples 'it supports tests with matching settings',
        'extra application settings in environment',
        '{"feature_enabled": true}',
        { 'feature_enabled' => true, 'extra_setting' => 'value', 'another_setting' => 5 }
    end

    describe 'when a setting in test @gitlab_settings tag that does not match the GitLab instance settings' do
      shared_examples 'it does not support tests with matching settings' do |description, required_settings, actual_settings|
        context "with #{description}" do
          let(:test_required_settings) { required_settings }
          let(:gitlab_env_settings) { actual_settings }

          it 'returns true (test is not supported)' do
            expect(result).to be false
          end
        end
      end

      include_examples 'it does not support tests with matching settings',
        'boolean true=false',
        '{"a": true}',
        { 'a' => false }

      include_examples 'it does not support tests with matching settings',
        'boolean false=true',
        '{"b": false}',
        { 'b' => true }

      include_examples 'it does not support tests with matching settings',
        'numeric value',
        '{"numeric_value": 101}',
        { 'numeric_value' => 100 }

      include_examples 'it does not support tests with matching settings',
        'string value',
        '{"string_value": "this-is-a-string"}',
        { 'string_value' => 'this-is-a-different-string' }

      include_examples 'it does not support tests with matching settings',
        'complex nested object',
        '{"config": {"a": true, "b": 5}}',
        { 'config' => { 'a' => true, 'b' => 4 } }

      include_examples 'it does not support tests with matching settings',
        'array',
        '{"allowed_values": ["a", "b", "c"]}',
        { 'allowed_values' => %w[a d c] }

      include_examples 'it does not support tests with matching settings',
        'complex nested object with missing keys in environment',
        '{"config": {"a": true, "b": 5, "c": "value"}}',
        { 'config' => { 'a' => true, 'b' => 5 } }

      include_examples 'it does not support tests with matching settings',
        'complex nested object with different structure',
        '{"config": {"a": true, "nested": {"x": 1}}}',
        { 'config' => { 'a' => true, 'nested' => 'not an object' } }
    end

    describe 'when a setting in test @gitlab_settings tag does not exist in the GitLab instance settings' do
      let(:gitlab_env_settings) { { 'existing_setting' => true } }

      context 'when the test requires the non-existent setting to be false' do
        let(:test_required_settings) { '{"non_existent_setting": false}' }

        it 'allows the test to run (backward compatible)' do
          expect(result).to be true
        end
      end

      context 'when the test requires the non-existent setting to be 0' do
        let(:test_required_settings) { '{"non_existent_setting": 0}' }

        it 'allows the test to run (backward compatible)' do
          expect(result).to be true
        end
      end

      context 'when the test requires the non-existent setting to be null' do
        let(:test_required_settings) { '{"non_existent_setting": null}' }

        it 'allows the test to run (backward compatible)' do
          expect(result).to be true
        end
      end

      context 'when the test requires the non-existent setting to be true' do
        let(:test_required_settings) { '{"non_existent_setting": true}' }

        it 'skips the test (feature required but not available)' do
          expect(result).to be false
        end
      end

      context 'when the test requires the non-existent setting to be a non-zero number' do
        let(:test_required_settings) { '{"non_existent_setting": 5}' }

        it 'skips the test (feature required but not available)' do
          expect(result).to be false
        end
      end

      context 'when the test requires the non-existent setting to be a string' do
        let(:test_required_settings) { '{"non_existent_setting": "value"}' }

        it 'skips the test (feature required but not available)' do
          expect(result).to be false
        end
      end
    end

    describe "rate limit handling" do
      let(:env_url) { "https://example.gitlab.com" }
      let(:test_file) { "path/to/api_test.js" }
      let(:gitlab_env_settings) { { "api_request_limit" => 99 } }
      let(:test_required_settings) { '{"api_request_limit": 100}' }
      let(:unattended) { false }

      before do
        allow(described_class).to receive(:disable_rate_limit_setting) do |env_url, setting, value|
          GPTCommon.change_env_settings(
            env_url: env_url, headers: { 'PRIVATE-TOKEN': 'test_token' }, settings: { setting.to_sym => value }
          )
        end
      end

      it "shows a warning prompt when not in unattended mode" do
        described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, true, unattended
        )

        expect(GPTCommon).to have_received(:show_warning_prompt).with(
          include("update the GitLab Environment 'api_request_limit' setting from 99 to 100")
        )
        expect(described_class).to have_received(:disable_rate_limit_setting)
      end

      it "doesn't show a warning prompt when in unattended mode" do
        described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, true, true
        )

        expect(GPTCommon).not_to have_received(:show_warning_prompt)
        expect(described_class).to have_received(:disable_rate_limit_setting)
      end

      it "calls change_env_settings with correct parameters" do
        described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, true, unattended
        )

        expect(GPTCommon).to have_received(:change_env_settings).with(
          env_url: env_url,
          headers: { 'PRIVATE-TOKEN': 'test_token' },
          settings: { api_request_limit: 100 }
        )
      end

      it "doesn't modify settings when rate_limits is false" do
        described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, false, unattended
        )

        expect(GPTCommon).not_to have_received(:change_env_settings)
      end

      context "when params are not rate limit related" do
        let(:gitlab_env_settings) { { "api_request_limit" => 99 } }
        let(:test_required_settings) { '{"api_request_value": 100}' } # logic expects setting name include 'limit'

        it "doesn't apply rate limit handling" do
          described_class.test_supported_by_gitlab_settings?(
            test_file, gitlab_env_settings, env_url, true, unattended
          )
          expect(GPTCommon).not_to have_received(:change_env_settings)
        end
      end

      it "returns true after successfully updating the rate limit" do
        result = described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, true, unattended
        )

        expect(result).to be true
      end

      it "logs information and calls change_env_settings when disabling rate limits" do
        allow(GPTLogger.logger).to receive(:info)
        allow(GPTCommon).to receive(:change_env_settings)
        allow(described_class).to receive(:disable_rate_limit_setting).and_call_original

        described_class.test_supported_by_gitlab_settings?(
          test_file, gitlab_env_settings, env_url, true, unattended
        )

        expect(GPTLogger.logger).to have_received(:info).with(
          "Updating 'api_request_limit' limit on environment to 100 during the test..."
        )
        expect(GPTCommon).to have_received(:change_env_settings).with(
          env_url: env_url,
          headers: { 'PRIVATE-TOKEN': 'test_token' },
          settings: { api_request_limit: 100 }
        )
      end
    end
  end
end
