require 'spec_helper'
require 'tmpdir'
require 'fileutils'
require_relative '../lib/run_k6'

RSpec.describe "K6 Integration" do
  let(:k6_path) { RunK6.setup_k6 }
  let(:minimal_test_content) do
    <<~JS
      import http from 'k6/http';
      import { Rate } from "k6/metrics";

      export let successRate = new Rate("successful_requests")
      export const options = {
         "stages": [
          { "duration": "3s", "target": 1 },
          { "duration": "1s", "target": 0 }
          ],
          "rps": 1,
          "batchPerHost": 0,
          thresholds: {
          'http_req_duration': ['p(90) < 1000'],
          'http_reqs': ['count >= 2'],
          'successful_requests': ['rate > 0.99'],
          "http_reqs{endpoint:custom_tag}": [`count>=2`],
        }
      };

      export function setup() {
        console.log("RPS Threshold: 1.0/s");
        console.log("TTFB P90 Threshold: 500ms");
        console.log(`Success Rate Threshold: 99%`)
      }

      export default function() {
        let params = { tags: { endpoint: 'custom_tag' } };
        const res = http.get('https://gitlab.com', params);
        successRate.add(res.status === 200);
      }
    JS
  end

  let(:test_dir) { Dir.mktmpdir }
  let(:results_dir) { File.join(test_dir, "results") }
  let(:minimal_test_file) { File.join(test_dir, "minimal_test.js") }

  before do
    FileUtils.mkdir_p(results_dir)
    File.write(minimal_test_file, minimal_test_content)
  end

  after do
    FileUtils.remove_entry(test_dir) if File.directory?(test_dir)
  end

  it "can run k6 and parse its output correctly" do
    env_vars = { 'ENVIRONMENT_NAME' => 'integration_test' }

    success, output = RunK6.run_k6(
      k6_path: k6_path,
      opts: {},
      env_vars: env_vars,
      options_file: nil,
      test_file: minimal_test_file,
      results_dir: results_dir,
      gpt_version: "test-version"
    )

    # Verify the test ran successfully
    expect(success).to be true
    expect(output).not_to be_empty
    expect(output.any? { |line| line.include?('data_received') }).to be true
    expect(output.any? { |line| line.include?('data_sent') }).to be true
    expect(output.any? { |line| line.include?('http_reqs') }).to be true
    expect(output.any? { |line| line.include?('iteration_duration') }).to be true
    expect(output.any? { |line| line.include?('endpoint:custom_tag') }).to be true

    # Parse results
    results = RunK6.get_test_results(
      test_file: minimal_test_file,
      status: success,
      output: output,
      test_redo: false
    )

    # Output the parsed results for debugging
    puts "Parsed Results:"
    puts results.inspect

    # Verify critical metrics were extracted
    expect(results["result"]).to be true
    expect(results["ttfb_avg"]).to be_a(String)
    expect(results["ttfb_p90"]).to be_a(String)
    expect(results["rps_result"]).to be_a(String)
    expect(results["success_rate"]).to eq("100.00")
  end
end
