require 'spec_helper'
require 'rainbow'
require_relative '../lib/gpt_common'

RSpec.describe GPTCommon do
  before do
    allow(GPTLogger.logger).to receive(:info)
    allow(GPTLogger.logger).to receive(:warn)

    stub_const("ENV", ENV.to_hash.merge({
      'ACCESS_TOKEN' => 'test-token'
    }))
  end

  describe '.make_http_request' do
    let(:url) { 'https://gitlab.example.com/api' }
    let(:headers) { { 'Content-Type' => 'application/json' } }

    context 'with successful requests' do
      before do
        stub_request(:get, url)
          .to_return(status: 200, body: '{"success": true}', headers: { 'Content-Type' => 'application/json' })
      end

      it 'returns response for successful requests' do
        response = described_class.make_http_request(method: 'get', url: url, headers: headers)
        expect(response.status.success?).to be true
        expect(JSON.parse(response.body.to_s)).to eq({ 'success' => true })
      end

      it 'logs response body when show_response is true' do
        expect(GPTLogger.logger).to receive(:info).with({ 'success' => true })
        described_class.make_http_request(method: 'get', url: url, headers: headers, show_response: true)
      end
    end

    context 'with error handling' do
      it 'raises error when URL is missing' do
        expect do
          described_class.make_http_request(method: 'get', url: nil)
        end.to raise_error(/URL not defined/)
      end

      it 'includes correlation ID in error message' do
        stub_request(:get, url)
          .to_return(status: 422, body: '{"error": "invalid"}',
            headers: { 'x-request-id' => 'abc123', 'Content-Type' => 'application/json' })

        expect do
          described_class.make_http_request(method: 'get', url: url, headers: headers)
        end.to raise_error(GPTCommon::RequestError, /Correlation ID: abc123/)
      end

      it 'includes rate limit name in error message' do
        stub_request(:get, url)
          .to_return(status: 429, body: '{"error": "rate limited"}',
            headers: { 'ratelimit-name' => 'api-calls', 'Content-Type' => 'application/json' })

        expect do
          described_class.make_http_request(method: 'get', url: url, headers: headers)
        end.to raise_error(GPTCommon::RequestError, /Rate Limit error caused by 'api-calls' limit/)
      end

      it 'retries on 502 errors when configured' do
        # First call returns 502, second call succeeds
        stub_request(:get, url)
          .to_return({ status: 502, body: 'Bad Gateway' },
            { status: 200, body: '{"success": true}', headers: { 'Content-Type' => 'application/json' } })

        allow(described_class).to receive(:sleep)

        response = described_class.make_http_request(method: 'get', url: url, headers: headers, retry_on_error: true)
        expect(response.status.success?).to be true
      end

      it 'raises BadGatewayError after retry fails' do
        stub_request(:get, url).to_return(status: 502, body: 'Bad Gateway')

        allow(described_class).to receive(:sleep)

        expect do
          described_class.make_http_request(method: 'get', url: url, headers: headers, retry_on_error: true)
        end.to raise_error(GPTCommon::BadGatewayError)
      end

      it 'does not retry when retry_on_error is false' do
        stub_request(:get, url).to_return(status: 502, body: 'Bad Gateway')

        expect do
          described_class.make_http_request(method: 'get', url: url, headers: headers, retry_on_error: false)
        end.to raise_error(GPTCommon::BadGatewayError)
      end

      it 'does not raise error when fail_on_error is false' do
        stub_request(:get, url).to_return(status: 404, body: '{"error": "not found"}')

        response = described_class.make_http_request(method: 'get', url: url, headers: headers, fail_on_error: false)
        expect(response.status).to eq(404)
      end
    end

    context 'with body parameter' do
      let(:body) { '{"data": "test"}' }

      before do
        stub_request(:post, url)
          .with(body: body, headers: headers)
          .to_return(status: 201, body: '{"id": 123}', headers: { 'Content-Type' => 'application/json' })
      end

      it 'sends the body with the request' do
        response = described_class.make_http_request(method: 'post', url: url, body: body, headers: headers)
        expect(response.status).to eq(201)
        expect(JSON.parse(response.body.to_s)).to eq({ 'id' => 123 })
      end
    end

    context 'with follow_redirects' do
      before do
        stub_request(:get, url).to_return(status: 200, body: '{"success": true}')
      end

      it 'follows redirects when true' do
        expect(HTTP).to receive(:follow).and_call_original
        described_class.make_http_request(method: 'get', url: url, follow_redirects: true)
      end

      it 'does not follow redirects when false' do
        expect(HTTP).not_to receive(:follow)
        described_class.make_http_request(method: 'get', url: url, follow_redirects: false)
      end
    end
  end

  describe '.download_file' do
    let(:url) { 'https://example.com/file.zip' }
    let(:tempfile) { instance_double(Tempfile) }

    it 'downloads file successfully' do
      allow(Down).to receive(:download).with(url).and_return(tempfile)
      expect(described_class.download_file(url: url)).to eq(tempfile)
    end

    it 'uses proxy when configured' do
      stub_const("ENV", ENV.to_hash.merge({ 'PROXY_URL' => 'http://proxy:8080' }))
      expect(Down).to receive(:download).with(url, proxy: 'http://proxy:8080').and_return(tempfile)
      described_class.download_file(url: url)
    end

    it 'raises meaningful error on timeout' do
      timeout_error = Down::TimeoutError.new
      allow(Down).to receive(:download).and_raise(timeout_error)

      expect do
        described_class.download_file(url: url)
      end.to raise_error(Down::TimeoutError, /timed out/)
    end

    it 'raises meaningful error on download failure' do
      download_error = Down::Error.new("Connection refused")
      allow(Down).to receive(:download).and_raise(download_error)

      expect do
        described_class.download_file(url: url)
      end.to raise_error(Down::Error, /Connection refused/)
    end
  end

  describe '.check_gitlab_env_and_token' do
    let(:env_url) { 'https://gitlab.example.com' }
    let(:version_url) { "#{env_url}/api/v4/version" }

    it 'validates GitLab environment and token' do
      stub_request(:get, version_url)
        .with(headers: { 'PRIVATE-TOKEN' => 'test-token' })
        .to_return(status: 200, body: '{"version": "14.5.0"}')

      version = described_class.check_gitlab_env_and_token(env_url: env_url)
      expect(version.to_s).to eq('14.5.0')
    end

    it 'warns about older supported versions' do
      stub_request(:get, version_url).to_return(status: 200, body: '{"version": "12.4.0"}')

      expect(GPTLogger.logger).to receive(:warn)
        .with(matching(/Target environment is 12.4.0. Minimum fully supported GitLab version/))
      described_class.check_gitlab_env_and_token(env_url: env_url)
    end

    it 'raises error for unsupported versions' do
      stub_request(:get, version_url)
        .to_return(status: 200, body: '{"version": "10.9.9"}')

      expect { described_class.check_gitlab_env_and_token(env_url: env_url) }
        .to raise_error(/versions lower than 11.0.0 are unsupported/)
    end

    it 'raises error when environment cannot be reached' do
      stub_request(:get, version_url)
        .to_return(status: 500, body: '{"error": "Internal Server Error"}',
          headers: { 'Content-Type' => 'application/json' })

      expect { described_class.check_gitlab_env_and_token(env_url: env_url) }
        .to raise_error(/Environment access token check has failed/)
    end
  end

  describe '.get_env_settings' do
    let(:env_url) { 'https://gitlab.example.com' }
    let(:settings_url) { "#{env_url}/api/v4/application/settings" }

    it 'returns settings when successful' do
      stub_request(:get, settings_url)
        .with(headers: { 'PRIVATE-TOKEN' => 'test-token' })
        .to_return(status: 200, body: '{"setting1": "value1"}')

      settings = described_class.get_env_settings(env_url: env_url)
      expect(settings).to eq({ 'setting1' => 'value1' })
    end

    it 'returns empty hash when request fails' do
      stub_request(:get, settings_url)
        .to_return(status: 403, body: '{"error": "Forbidden"}')

      settings = described_class.get_env_settings(env_url: env_url)
      expect(settings).to eq({})
    end

    it 'returns false when access token is not set' do
      stub_const("ENV", ENV.to_hash.merge({ 'ACCESS_TOKEN' => nil }))

      settings = described_class.get_env_settings(env_url: env_url)
      expect(settings).to be false
    end
  end

  describe '.change_env_settings' do
    let(:env_url) { 'https://gitlab.example.com' }
    let(:settings_url) { "#{env_url}/api/v4/application/settings" }
    let(:headers) { { 'PRIVATE-TOKEN' => 'test-token' } }
    let(:settings) { { 'signup_enabled' => false } }

    it 'successfully changes environment settings' do
      stub_request(:put, settings_url)
        .with(body: { 'signup_enabled' => false }, headers: headers)
        .to_return(status: 200, body: '{"signup_enabled": false}')

      allow(described_class).to receive(:sleep)

      expect { described_class.change_env_settings(env_url: env_url, headers: headers, settings: settings) }
        .not_to raise_error
    end

    it 'raises error when settings change fails' do
      stub_request(:put, settings_url)
        .to_return(status: 403, body: '{"error": "Forbidden"}')

      expect { described_class.change_env_settings(env_url: env_url, headers: headers, settings: settings) }
        .to raise_error(/Request has failed/)
    end
  end

  describe '.get_license_plan' do
    let(:env_url) { 'https://gitlab.example.com' }
    let(:license_url) { "#{env_url}/api/v4/license" }

    it 'returns license plan when successful' do
      stub_request(:get, license_url)
        .with(headers: { 'PRIVATE-TOKEN' => 'test-token' })
        .to_return(status: 200, body: '{"plan": "Ultimate"}')

      plan = described_class.get_license_plan(env_url: env_url)
      expect(plan).to eq('ultimate')
    end

    it 'returns empty string when no license found' do
      stub_request(:get, license_url)
        .to_return(status: 200, body: 'null')

      plan = described_class.get_license_plan(env_url: env_url)
      expect(plan).to eq('')
    end

    it 'returns empty string when license check fails' do
      stub_request(:get, license_url)
        .to_return(status: 403, body: '{"error": "Forbidden"}')

      plan = described_class.get_license_plan(env_url: env_url)
      expect(plan).to eq('')
    end

    it 'returns empty string when ENVIRONMENT_LICENSE_CHECK_SKIP is set' do
      stub_const("ENV", ENV.to_hash.merge({ 'ENVIRONMENT_LICENSE_CHECK_SKIP' => 'true' }))

      plan = described_class.get_license_plan(env_url: env_url)
      expect(plan).to eq('')
    end
  end

  describe '.show_warning_prompt' do
    it 'continues when user enters Y' do
      allow($stdin).to receive(:gets).and_return("Y\n")
      expect { described_class.show_warning_prompt("Warning") }.not_to raise_error
    end

    it 'continues when user enters yes' do
      allow($stdin).to receive(:gets).and_return("yes\n")
      expect { described_class.show_warning_prompt("Warning") }.not_to raise_error
    end

    it 'aborts when user enters N' do
      allow($stdin).to receive(:gets).and_return("N\n")
      expect(described_class).to receive(:abort)
      described_class.show_warning_prompt("Warning")
    end

    it 'aborts when user enters yesno' do
      allow($stdin).to receive(:gets).and_return("yesno\n")
      expect(described_class).to receive(:abort)
      described_class.show_warning_prompt("Warning")
    end
  end
end
