$LOAD_PATH.unshift File.expand_path('.', __dir__)
require 'run_k6'
require 'semantic'

module TestInfo
  extend self

  # Utility
  def get_test_tag_value(test_file, tag)
    File.open(test_file, "r") do |test_file_content|
      test_file_content.each_line do |line|
        return line.match(/@#{tag}: (.*)\n/)[1] if line.match?(/@#{tag}:/)
      end
    end

    nil
  end

  # Get

  def get_tests_info(test_files)
    test_files = test_files.split(',') if test_files.instance_of?(String)
    info_list = []

    test_files.each do |test_file|
      info = {}
      info.default = nil

      info[:name] = File.basename(test_file, '.js')
      info[:type] = test_file.split("/")[-2]
      info[:link] = "https://gitlab.com/gitlab-org/quality/performance/blob/main/k6/tests/#{info[:type]}/#{info[:name]}.js"
      info[:link_md] = "[#{info[:name]}](#{info[:link]})"

      info[:description] = get_test_tag_value(test_file, 'description')
      info[:endpoint] = get_test_tag_value(test_file, 'endpoint') || 'No documentaion'
      info[:example_uri] = get_test_tag_value(test_file, 'example_uri')
      info[:issues] = get_test_tag_value(test_file, 'issue')
      info[:previous_issues] = get_test_tag_value(test_file, 'previous_issues')
      info[:gitlab_version] = get_test_tag_value(test_file, 'gitlab_version')
      info[:flags] = get_test_tag_value(test_file, 'flags')
      info[:gitlab_settings] = get_test_tag_value(test_file, 'gitlab_settings')
      info[:gpt_data_version] = get_test_tag_value(test_file, 'gpt_data_version')
      info[:stressed_components] = get_test_tag_value(test_file, 'stressed_components')

      info_list << info
    end

    info_list
  end

  def get_test_urls(tests_info, env_vars)
    env_url = env_vars['ENVIRONMENT_URL']
    large_project = JSON.parse(env_vars['ENVIRONMENT_LARGE_PROJECTS']).first
    horizontal_data = JSON.parse(env_vars['ENVIRONMENT_MANY_GROUPS_AND_PROJECTS'])
    additional_data = { "environment_root_group" => env_vars['ENVIRONMENT_ROOT_GROUP'], "user" => env_vars['ENVIRONMENT_USER'] }
    test_data = large_project.merge(horizontal_data, additional_data)

    tests_info.each do |test_info|
      if test_info[:example_uri].nil?
        test_info[:url] = ''
        next
      elsif !test_info[:example_uri].include?(':')
        test_info[:url] = "#{env_url}#{test_info[:example_uri]}"
        next
      end

      # Substitute all options like `:encoded_path` with test data from target env
      endpoint = test_info[:example_uri].gsub(/(\/|=):(\w+)/) do |match|
        test_option = match.gsub(/[^0-9A-Za-z_]/, '')
        if test_data[test_option].nil?
          test_info[:example_uri] = nil
          next
        else
          "/#{test_data[test_option]}"
        end
      end
      test_info[:url] = test_info[:example_uri].nil? ? '' : "#{env_url}#{endpoint}"
    end
    tests_info
  end

  # Check

  def test_has_unsafe_requests?(test_file)
    return true if test_has_flag?(test_file, 'unsafe')

    write_methods = %w[put del patch]
    File.open(test_file, "r") do |test_file_content|
      test_file_content.each_line do |line|
        line_has_write_method = write_methods.any? { |write_method| line.include?("http.#{write_method}") }
        return true if line_has_write_method
      end
    end

    false
  end

  # check if k6 test has a specific flag
  def test_has_flag?(test_file, flag)
    get_test_tag_value(test_file, 'flags')&.include?(flag)
  end

  def test_supported_by_gitlab_version?(test_file, gitlab_version)
    test_supported_version = get_test_tag_value(test_file, 'gitlab_version')
    return true if test_supported_version.nil?

    if test_supported_version && gitlab_version == '-'
      warn Rainbow("GitLab version wasn't able to be determined. Test '#{File.basename(test_file)}' requires GitLab version '#{test_supported_version}' and up. Check that the environment is accessible and the ACCESS_TOKEN provided is correct then try again. Skipping out of caution...").yellow
      return false
    end

    gitlab_version = Semantic::Version.new(gitlab_version.match(/\d+\.\d+\.\d+/)[0])
    if test_supported_version && gitlab_version < Semantic::Version.new(test_supported_version)
      warn Rainbow("Test '#{File.basename(test_file)}' isn't supported by target GitLab environment version '#{gitlab_version}'. Requires '#{test_supported_version}' and up. Skipping...").yellow
      return false
    end

    true
  end

  # - Checks if the test is supported by the GitLab settings in the target environment based on environment application settings
  # - For settings that are 'rate limit' related, can also disable the rate limit setting in the target environment
  #
  # @param [String] test_file - The path to the test file
  # @param [Hash{String => Object}] gitlab_env_settings - The settings of the target environment
  # @param [String] env_url - The URL of the target environment
  # @param [Boolean] rate_limits - Disable rate limit application settings in target environment
  # @param [Boolean] unattended - Skip all user prompts and runs automatically
  # @return [Boolean] - True if the test is supported by the GitLab settings in the target environment
  def test_supported_by_gitlab_settings?(test_file, gitlab_env_settings, env_url, rate_limits, unattended)
    test_name = File.basename(test_file)

    test_required_settings = get_test_tag_value(test_file, 'gitlab_settings')
    return true if test_required_settings.nil?
    return false if gitlab_env_settings.nil? || gitlab_env_settings.empty?

    begin
      parsed_settings = JSON.parse(test_required_settings)
      check_settings_compatibility(parsed_settings, gitlab_env_settings, test_name, env_url, rate_limits, unattended)
    rescue StandardError => e
      GPTLogger.logger.error "Failed to parse JSON #{e}"
      false
    end
  end

  def disable_rate_limit_setting(env_url, rate_limit_setting, value)
    GPTLogger.logger.info "Updating '#{rate_limit_setting}' limit on environment to #{value} during the test..."
    GPTCommon.change_env_settings(env_url:, headers: { 'PRIVATE-TOKEN': ENV['ACCESS_TOKEN'] }, settings: { rate_limit_setting.to_sym => value })
  end

  def test_supported_by_gpt_data?(test_file, gpt_data_version)
    test_required_gpt_data = get_test_tag_value(test_file, 'gpt_data_version')
    return false if test_required_gpt_data.nil? || test_required_gpt_data.empty?

    if test_required_gpt_data && gpt_data_version == '-'
      warn Rainbow("GPT test data version wasn't able to be determined. Test '#{File.basename(test_file)}' requires GPT test data version '#{test_required_gpt_data}' and up. Check that the test data was setup correctly using the GPT Data Generator.\nIf you are running against a custom large project please disable this version check by adding `\"skip_check_version\": \"true\"` under `gpt_data` in Environment Config file.\nSkipping...").yellow
      return false
    end

    if test_required_gpt_data > gpt_data_version
      warn Rainbow("Test '#{File.basename(test_file)}' isn't supported by GPT test data version '#{gpt_data_version}' on the target GitLab environment. Requires '#{test_required_gpt_data}' and up. Please update test data using the latest GPT Data Generator. Skipping...").yellow
      return false
    end

    true
  end

  private

  def check_settings_compatibility(parsed_settings, gitlab_env_settings, test_name, env_url, rate_limits, unattended)
    parsed_settings.each do |setting, value|
      # Skip if setting is missing but value is falsey (backward compatibility)
      if gitlab_env_settings[setting].nil?
        next if falsey_value?(value)

        warn_unsupported_version(test_name, setting)
        return false
      end

      next if gitlab_env_settings[setting] == value

      # Handle rate limit settings
      if rate_limits && setting.match?(/limit/i)
        handle_rate_limit(setting, value, gitlab_env_settings, env_url, test_name, unattended)
        next
      end

      warn_value_mismatch(test_name, setting, value)
      return false
    end

    true
  end

  def falsey_value?(value)
    (value.is_a?(Numeric) && value.zero?) || value == false || value.nil?
  end

  def warn_unsupported_version(test_name, setting)
    warn Rainbow(
      "Test '#{test_name}' isn't supported by target GitLab environment due to " \
        "required environment setting '#{setting}' not being supported in this version. Skipping..."
    ).yellow
  end

  def warn_value_mismatch(test_name, setting, value)
    warn Rainbow(
      "Test '#{test_name}' isn't supported by target GitLab environment due to " \
        "required environment setting '#{setting}' not being set to '#{value}'. Skipping..."
    ).yellow
  end

  def handle_rate_limit(setting, value, gitlab_env_settings, env_url, test_name, unattended)
    unless unattended
      GPTCommon.show_warning_prompt(
        "GPT will update the GitLab Environment '#{setting}' setting from " \
          "#{gitlab_env_settings[setting]} to #{value} to allow for #{test_name} to run.\n" \
          "While the GPT is running this setting change will be in effect.\n" \
          "The original setting will be restored at the end of test run.\n" \
          "Note: this prompt can be skipped with `unattended` option."
      )
    end

    disable_rate_limit_setting(env_url, setting, value)
    gitlab_env_settings[setting] = value # to skip updating same setting for other tests with the same rate limit
  end
end
