module ResultWriter
  class Base
    attr_reader :path

    def self.sorted_results(results)
      return results unless ENV['GPT_RESULTS_SORT_BY_FAILED']

      # Sort by result - failed then passed, then sort tests alphabetically within each group (failed and passed)
      results.sort_by { |test_result| [test_result['result'] ? 1 : 0, test_result['name']] }
    end

    def initialize(results_dir, file_prefix)
      ext = self.class.name.split('::').last.downcase
      @path = File.join(results_dir, "#{file_prefix}_results.#{ext}")
    end
  end

  class Json < Base
    def write(results)
      File.write(path, results.to_json)
    end
  end

  class Csv < Base
    require 'csv'

    def write(results)
      CSV.open(path, "wb") do |csv|
        csv << ["Name", "RPS", "RPS Result", "RPS Threshold", "TTFB Avg (ms)", "TTFB P90 (ms)", "TTFB P90 Threshold (ms)", "TTFB P95 (ms)", "Req Status", "Req Status Threshold", "Result"]
        Base.sorted_results(results['test_results']).each do |test_result|
          csv << [
            test_result['name'],
            test_result['rps_target'],
            test_result['rps_result'],
            test_result['rps_threshold'],
            test_result['ttfb_avg'],
            test_result['ttfb_p90'],
            test_result['ttfb_p90_threshold'],
            test_result['ttfb_p95'],
            test_result['success_rate'],
            test_result['success_rate_threshold'],
            test_result['result'] ? "Passed" : "FAILED"
          ]
        end
      end
    end
  end

  class Txt < Base
    require 'table_print'

    class Sections
      def generate(results)
        {
          summary: generate_results_summary(results),
          table: generate_results_table(results),
          footer: generate_results_footer(results)
        }
      end

      private

      def generate_results_summary(results)
        results_summary = <<~DOC
          * Environment:                #{results['name'].capitalize}
          * Environment Version:        #{results['version']} `#{results['revision']}`
          * Option:                     #{results['option']}
          * Date:                       #{results['date']}
          * Run Time:                   #{ChronicDuration.output(results['time']['run'], format: :short, keep_zero: true)} (Start: #{results['time']['start']}, End: #{results['time']['end']})
          * GPT Version:                v#{results['gpt_version']}
        DOC
        results_summary += "\n❯ Overall Results Score: #{results['overall_result_score']}%\n" unless results['overall_result_score'].nil?
        results_summary
      end

      def generate_results_table(results)
        tp_results = Base.sorted_results(results['test_results']).map do |test_result|
          tp_result = {}

          tp_result["Name"] = test_result['name'] || '-'
          tp_result["RPS"] = test_result['rps_target'] ? "#{test_result['rps_target']}/s" : '-'
          tp_result["RPS Result"] = [test_result['rps_result'], test_result['rps_threshold']].none?(&:nil?) ? "#{test_result['rps_result']}/s (>#{test_result['rps_threshold']}/s)" : '-'
          tp_result["TTFB Avg"] = test_result['ttfb_avg'] ? "#{test_result['ttfb_avg']}ms" : '-'
          tp_result["TTFB P90"] = [test_result['ttfb_p90'], test_result['ttfb_p90_threshold']].none?(&:nil?) ? "#{test_result['ttfb_p90']}ms (<#{test_result['ttfb_p90_threshold']}ms)" : '-'
          tp_result["TTFB P95"] = "#{test_result['ttfb_p95']}ms" if ENV['GPT_TTFB_P95']
          tp_result["Data Received"] = (test_result['data_received'] ? "#{test_result['data_received'][1]} (#{test_result['data_received'][2]})" : '-') if ENV['GPT_DATA_RECEIVED']
          tp_result["Req Status"] = [test_result['success_rate'], test_result['success_rate_threshold']].none?(&:nil?) ? "#{test_result['success_rate']}% (>#{test_result['success_rate_threshold']}%)" : '-'

          test_result_str = test_result['result'] ? "Passed" : "FAILED"
          test_result_str << '¹' unless test_result['issues'].nil?
          test_result_str << '²' unless test_result['result']
          tp_result["Result"] = test_result['result'] ? Rainbow(test_result_str).green : Rainbow(test_result_str).red

          tp_result
        end

        tp.set(:max_width, 60)
        TablePrint::Printer.table_print(tp_results)
      end

      def generate_results_footer(results)
        footer = ''
        footer << "\n¹ Result covers endpoint(s) that have known issue(s). Threshold(s) have been adjusted to compensate." if results['test_results'].any? { |test_result| test_result['issues'] }
        footer << "\n² Failure may not be clear from summary alone. Refer to the individual test's full output for further debugging." unless results['overall_result']
        footer
      end
    end

    def write(results, skip_posting_summary = false)
      sections = Sections.new.generate(results)
      summary, table, footer = sections.values_at(:summary, :table, :footer)
      # Write results to file but also remove any terminal ANSI codes
      summary_results = "#{summary}\n#{table}\n#{footer}".gsub(/\e\[([;\d]+)?m/, '')
      File.write(path, summary_results)

      return if skip_posting_summary

      puts "\n█ Results summary\n\n#{summary}\n"
      puts table
      puts Rainbow(footer).italic unless footer.empty?
    end
  end

  class MD < Base
    def write(results)
      report = generate_failure_evaluation_report(results)
      File.write(@path, report)
    end

    def add_failure_evaluation_data(results)
      results["evaluation"] = {}
      return results["evaluation"] if results["result"]

      small_failure_margin = 10
      failure_percentages = calculate_percent_diff_from_threshold(results)

      evaluations = {
        small_failure: "Test failed by a small amount within #{small_failure_margin}% of the threshold. Rerun the test to check for consistency.",
        ttfb: "`TTFB P90` failure - #{results['ttfb_p90']}ms (Threshold: #{results['ttfb_p90_threshold']}ms) - check CPU/Memory usage and GitLab logs for request durations and queue times.",
        rps: "`RPS RESULT` failure - #{results['rps_result']}/s (Threshold: #{results['rps_threshold']}/s) - check CPU/Memory usage and GitLab logs for request durations and queue times.",
        req_status: "`REQ STATUS` failure - #{results['success_rate']}% (Threshold: #{results['success_rate_threshold']}%) - check `_results_output.log` for response codes and Correlation IDs. Use these to trace errors in GitLab logs.",
        web_failures: "Web test failure with no visible failed threshold - check `_results_output.log` file for full details which endpoint failed in Web test by searching for test name."
      }

      failed_thresholds = failure_percentages.values.select(&:positive?)
      results["evaluation"]["small_failure"] = evaluations[:small_failure] if !failed_thresholds.empty? && failed_thresholds.all? { |v| v <= small_failure_margin }

      results["evaluation"]["ttfb"] = evaluations[:ttfb] if failure_percentages[:ttfb_p90] > small_failure_margin
      results["evaluation"]["rps"] = evaluations[:rps] if failure_percentages[:rps] > small_failure_margin
      results["evaluation"]["req_status"] = evaluations[:req_status] if failure_percentages[:success_rate] > small_failure_margin
      results["evaluation"]["web_failures"] = evaluations[:web_failures] if results["name"].include?('web') && failed_thresholds.empty?

      results["evaluation"]["error"] = "Error occurred during the test: `#{results['error']}`" if results["error"]
      results["evaluation"]["rate_limit_error"] = "Rate limit error occurred during the test: `#{results['rate_limit_error']}`" if results["rate_limit_error"]
      results["evaluation"]["threshold_error"] = "Threshold violation during the test: `#{results['threshold_error']}`" if results["threshold_error"]

      results["evaluation"]
    end

    def calculate_percent_diff_from_threshold(results)
      {
        rps: calculate_diff_percentage(results["rps_result"], results["rps_threshold"], lower_is_worse: true),
        ttfb_p90: calculate_diff_percentage(results["ttfb_p90"], results["ttfb_p90_threshold"], lower_is_worse: false),
        success_rate: calculate_diff_percentage(results["success_rate"], results["success_rate_threshold"], lower_is_worse: true)
      }
    end

    def calculate_diff_percentage(actual, threshold, lower_is_worse:)
      actual = actual.to_f
      threshold = threshold.to_f
      # When threshold not violated, set failure diff to 0
      return 0 if (lower_is_worse && actual >= threshold) || (!lower_is_worse && actual <= threshold)

      ((actual - threshold).abs / threshold * 100).round(2)
    end

    def generate_failure_evaluation_report(results)
      report = <<~DOC
        # Failure Evaluation Report\n
        Automated evaluation of individual test failures.

        △ Please note:
        - This report analyzes failures in isolation and may not capture the full complexity of performance issues.
        - Comprehensive analysis requires examining patterns across all tests, system-wide metrics, and full review.
        - Use this as a starting point for investigation, not as a definitive diagnosis of performance problems.
      DOC

      results['test_results'].each do |test|
        next if test['result'] # Skip if the test passed

        test['evaluation'] = add_failure_evaluation_data(test)

        report += "\n## `#{test['name']}`\n\n"
        report += "Failure notes:\n"

        test['evaluation'].each do |key, value|
          report += "- #{key}: #{value}\n"
        end

        if test['evaluation'].key?('ttfb') || test['evaluation'].key?('rps')
          report += "\nKey GitLab components typically stressed during the test: #{test['stressed_components']}\n" if test['stressed_components']
          report += "\nNote: endpoint has a known performance issue #{test['issues']}\n" if test['issues']
        end
      end

      report
    end
  end
end
