/*global __ENV : true  */
/*
@endpoint: `GET /:group/:project.git/info/refs?service=git-upload-pack`
@example_uri: /:unencoded_path.git/info/refs?service=git-upload-pack
@description: Git Refs List via HTTPS. <br>Controllers: `Repositories::GitHttpController#info_refs`</br>
@gpt_data_version: 1
@stressed_components: Gitaly, Rails
*/

import http from "k6/http";
import { group } from "k6";
import { Rate } from "k6/metrics";
import { logError, getRpsThresholds, getTtfbThreshold, getLargeProjects, selectRandom, adjustRps, adjustStageVUs } from "../../lib/gpt_k6_modules.js";

export let gitProtoRps = adjustRps(__ENV.GIT_PULL_ENDPOINT_THROUGHPUT)
export let gitProtoStages = adjustStageVUs(__ENV.GIT_PULL_ENDPOINT_THROUGHPUT)
export let rpsThresholds = getRpsThresholds(__ENV.GIT_PULL_ENDPOINT_THROUGHPUT)
export let ttfbThreshold = getTtfbThreshold()
export let successRate = new Rate("successful_requests")
export let options = {
  thresholds: {
    "successful_requests": [`rate>${__ENV.SUCCESS_RATE_THRESHOLD}`],
    "http_req_waiting": [`p(90)<${ttfbThreshold}`],
    "http_reqs": [`count>=${rpsThresholds['count']}`]
  },
  scenarios: {
    gpt: {
      executor: 'ramping-arrival-rate',
      preAllocatedVUs: gitProtoRps,
      maxVUs: __ENV.OPTION_MAX_VUS,
      stages: gitProtoStages,
      gracefulStop: '1800s',
    },
  },
  rps: gitProtoRps,
  discardResponseBodies: true // Configure k6 not to load response body with repository to memory
};

export let projects = getLargeProjects(['name', 'unencoded_path']);

export function setup() {
  console.log('')
  console.log(`Git Protocol RPS: ${gitProtoRps}`)
  console.log(`RPS Threshold: ${rpsThresholds['mean']}/s (${rpsThresholds['count']})`)
  console.log(`TTFB P90 Threshold: ${ttfbThreshold}ms`)
  console.log(`Success Rate Threshold: ${parseFloat(__ENV.SUCCESS_RATE_THRESHOLD) * 100}%`)
}

export default function () {
  group("Git - Get Refs List", function () {
    let project = selectRandom(projects);

    let params = { headers: { "Accept": "*/*", "Accept-Encoding": "deflate, gzip", "Pragma": "no-cache" }, timeout: "600s" };
    let res = http.get(`${__ENV.ENVIRONMENT_URL}/${project['unencoded_path']}.git/info/refs?service=git-upload-pack`, params);
    /20(0|1)/.test(res.status) ? successRate.add(true) : (successRate.add(false), logError(res));
  });
}
