/*global __ENV : true  */
/*
@endpoint: `POST /api/graphql`
@description: List project vulnerabilities based on severity filters
@gpt_data_version: 1
@flags: vulnerabilities
@issue: https://gitlab.com/gitlab-org/gitlab/-/issues/376142
@stressed_components: Rails, Postgres
*/

import http from "k6/http";
import { group } from "k6";
import { Rate } from "k6/metrics";

import { getRpsThresholds, getTtfbThreshold, selectRandom, getVulnerabilitiesProjects, logGraphqlError } from "../../lib/gpt_k6_modules.js";

export let thresholds = {
  'rps': { 'latest': 0.5 },
  'ttfb': { 'latest': 1800 }
};

export let rpsThresholds = getRpsThresholds(thresholds['rps'])
export let ttfbThreshold = getTtfbThreshold(thresholds['ttfb'])
export let successRate = new Rate("successful_requests")
export const projects = getVulnerabilitiesProjects()
export let projectToUse = projects["unencoded_paths"][0]

export let options = {
  thresholds: {
    "successful_requests": [`rate>${__ENV.SUCCESS_RATE_THRESHOLD}`],
    "http_req_waiting": [`p(90)<${ttfbThreshold}`],
    "http_reqs": [`count>=${rpsThresholds['count']}`]
  }
};

export function setup() {
  console.log('')
  console.log(`RPS Threshold: ${rpsThresholds['mean']}/s (${rpsThresholds['count']})`)
  console.log(`TTFB P90 Threshold: ${ttfbThreshold}ms`)
  console.log(`Success Rate Threshold: ${parseFloat(__ENV.SUCCESS_RATE_THRESHOLD)*100}%`)
}



export default function() {
  group("API - Projects Vulnerabilities List with Severity filters", function() {
    let headers = {
      "PRIVATE-TOKEN": `${__ENV.ACCESS_TOKEN}`,
      "Content-Type": "application/json"
    };
    let severity = selectRandom(["MEDIUM", "INFO", "UNKNOWN", "HIGH", "LOW", "CRITICAL"])
    let query = `
      query
        {
          project(fullPath: "${projectToUse}") {
            vulnerabilities(state: [DETECTED, CONFIRMED], severity: ${severity}, reportType: [API_FUZZING, CONTAINER_SCANNING, COVERAGE_FUZZING, DAST, DEPENDENCY_SCANNING, SAST, SECRET_DETECTION, GENERIC], first: 20, sort: severity_desc) {
                nodes {
                        id
                        title
                        state
                        severity
                        detectedAt
                        vulnerabilityPath
                        resolvedOnDefaultBranch
                        userNotesCount
                        issueLinks {
                          nodes {
                            id
                            issue {
                              id
                              iid
                              webUrl
                              webPath
                              title
                              state
                            }
                          }
                        }
                        identifiers {
                          externalType
                          name
                        }
                        project {
                          id
                          nameWithNamespace
                        }
                        reportType
                        scanner {
                          id
                          vendor
                        }
                        hasSolutions
                        mergeRequest {
                          id
                          webUrl
                          state
                          securityAutoFix
                          iid
                        }
                }
            }
          }
        }`;

    let res = http.post(`${__ENV.ENVIRONMENT_URL}/api/graphql`, JSON.stringify({ query: query }), {headers: headers});
    const graphQLErrors = JSON.parse(res.body).errors;
    graphQLErrors ? (successRate.add(false), logGraphqlError(graphQLErrors)) : successRate.add(true);
  });
}
