/*global __ENV : true  */
/*
@endpoint: `POST /api/graphql`
@description: List group vulnerabilities default frontend query
@gpt_data_version: 1
@flags: vulnerabilities
@issue: https://gitlab.com/gitlab-org/gitlab/-/issues/376142
@stressed_components: Rails, Postgres
*/

import http from "k6/http";
import { group } from "k6";
import { Rate } from "k6/metrics";

import {
  getRpsThresholds,
  getTtfbThreshold,
  getVulnerabilitiesGroup,
  logGraphqlError
} from "../../lib/gpt_k6_modules.js";

export let thresholds = {
  'ttfb': { 'latest': 1000 },
};

export let rpsThresholds = getRpsThresholds()
export let ttfbThreshold = getTtfbThreshold(thresholds['ttfb'])
export let successRate = new Rate("successful_requests")
export const groupToUse = getVulnerabilitiesGroup()


export let options = {
  thresholds: {
    "successful_requests": [`rate>${__ENV.SUCCESS_RATE_THRESHOLD}`],
    "http_req_waiting": [`p(90)<${ttfbThreshold}`],
    "http_reqs": [`count>=${rpsThresholds['count']}`]
  }
};

export function setup() {
  console.log('')
  console.log(`RPS Threshold: ${rpsThresholds['mean']}/s (${rpsThresholds['count']})`)
  console.log(`TTFB P90 Threshold: ${ttfbThreshold}ms`)
  console.log(`Success Rate Threshold: ${parseFloat(__ENV.SUCCESS_RATE_THRESHOLD)*100}%`)
}

export default function() {
  group("API - Groups Vulnerabilities List Default Query", function() {
    let headers = {
      "PRIVATE-TOKEN": `${__ENV.ACCESS_TOKEN}`,
      "Content-Type": "application/json"
    };
    let query = `
      query
        {
          group(fullPath: "${groupToUse}") {
            vulnerabilities(state: [DETECTED, CONFIRMED], reportType: [API_FUZZING, CONTAINER_SCANNING, COVERAGE_FUZZING, DAST, DEPENDENCY_SCANNING, SAST, SECRET_DETECTION, GENERIC], first: 20, sort: severity_desc) {
                nodes {
                        id
                        title
                        state
                        severity
                        detectedAt
                        vulnerabilityPath
                        resolvedOnDefaultBranch
                        userNotesCount
                        issueLinks {
                          nodes {
                            id
                            issue {
                              id
                              iid
                              webUrl
                              webPath
                              title
                              state
                            }
                          }
                        }
                        identifiers {
                          externalType
                          name
                        }
                         location{
                          __typename
                        }
                        project {
                          id
                          nameWithNamespace
                        }
                        reportType
                        scanner {
                          id
                          vendor
                        }
                }
            }
          }
        }`;

    let res = http.post(`${__ENV.ENVIRONMENT_URL}/api/graphql`, JSON.stringify({ query: query }), {headers: headers});
    const graphQLErrors = JSON.parse(res.body).errors;
    graphQLErrors ? (successRate.add(false), logGraphqlError(graphQLErrors)) : successRate.add(true);
  });
}
