# GitLab Performance Tool - Running the Tests

* [GitLab Performance Tool - Preparing the Environment](environment_prep.md)
* [**GitLab Performance Tool - Running the Tests**](k6.md)

With your [environment now prepared](environment_prep.md), next we'll detail how to configure and run [k6](https://k6.io/) tests against a GitLab environment with the GitLab Performance Tool (gpt).

**Note: Before running any tests with the Tool, the intended GitLab environment should be prepared first. Details on how to do this can be found here: [GitLab Performance Tool - Preparing the Environment](environment_prep.md)**

[[_TOC_]]

## Tool Requirements

### Software

GPT supports most popular Linux distributions and macOS.

GPT officially supports [supported GitLab versions](https://about.gitlab.com/support/statement-of-support/#version-support). While it may function with older GitLab versions, these are no longer officially supported.

### Hardware

The minimum hardware requirements for running the Tool are dependent on the intended load target (given as Requests per Second or RPS), the underlying hardware and the complexity of the tests being run.

We recommend that to run the default tests at **200 RPS** the machine running the Tool has approximately **1 CPU Core** and **1 GB RAM** available. So for example, to run the default tests at the following RPS targets the machine should have these specs available as a minimum:

* 200 RPS - 1 Core CPU, 1 GB RAM
* 500 RPS - 2 Core CPU, 2.5 GB RAM
* 1000 RPS - 4 Core CPU, 5 GB RAM

Both x86 and ARM are supported when running natively on Linux \ Mac and x86 for Docker. We generally recommend running the tool with [Docker](https://www.docker.com/) but the tool can also be run on Linux natively. More details can be found in the [Running the Tests with the Tool](#running-the-tests-with-the-tool) section below.

### Location and Network conditions

Two factors that can significantly affect the Tool's results are the conditions it's run in - Namely Location (physically in relation to the GitLab Environment) and Network.

GPT is designed to test the GitLab application's server performance directly at max throughputs, which in turn involves heavy network use. As such, **it's recommended to be run as close as possible physically to the GitLab environment and in optimum network conditions**. Through this, the results given by the Tool can be trusted to be representative of the application's actual performance and not disrupted by location or network issues. If this isn't possible the tool can be configured to specify a latency that it will take into account. See the [Environments](#environments) section for more info.

## Configuring the Tool

Before you can run performance tests with the tool against your GitLab environment you need to configure it. This involves configuring the tool with all the required info about your environment, the project(s) you intend to test against, any additional tests you wish to run and optionally to run the tests.

There are three key pieces of configuration in the tool - [Environments](../k6/config/environments), [Options](../k6/config/options) and [Tests](../k6/tests). This section details each as follows.

Out of the box all the default k6 tests are configured to run against an environment that has at least one instance of the default Test Project setup (see [GitLab Performance Tool - Preparing the Environment](environment_prep.md) for more info). However, this is configurable and the tool can be configured to run against any project providing it has equivalent data to test (more details in the next section).

### Environments

The first piece of configuration is the [Environment Config File](../k6/config/environments), which should already be prepared as part of the [Preparing the Environment](environment_prep.md##preparing-the-environment-file) steps. Refer back to that section in the docs if this file and the target GitLab environment isn't already prepared.

### Options (RPS)

The second piece of configuration is the [Options Config File](../k6/config/options). This file will set how the tool will run the tests - e.g. how long to run the tests for, how many users and how much throughput.

The [Options Config Files](../k6/config/options) are themselves native [k6 config files](https://grafana.com/docs/k6/latest/using-k6/k6-options). For this tool, we use them to set options, but they can also be used to set any valid k6 options as required for advanced use cases.

As an example, the following is one of our Options Config Files, [`20s_2rps.json`](../k6/config/options/20s_2rps.json), that configures the tests to each run for 20 seconds at a rate of 2 Requests Per Second (RPS):

```json
{
  "stages": [
    { "duration": "5s", "target": 2 },
    { "duration": "10s", "target": 2 },
    { "duration": "5s", "target": 0 }
  ],
  "rps": 2,
  "max_vus": 1000,
  "batchPerHost": 0
}
```

* `stages` - Defines the stages k6 should run the tests with. Sets the duration of each stage and how many users (VUs) to use.
  * It should be noted that each stage will ramp up from the previous, so in this example the scenario is to ramp up from 0 to 2 users over 5 seconds and then maintain 2 users for another 10s. Finally, it ramps down to 0 users in the next 5 seconds.
* `rps` - Sets the maximum Requests per Second that k6 can make in total.
* `max_vus` - Some tests may increase the amount of virtual users (VUs) as required in certain situations, such as to prevent the test slowing down if all users are tagged downloading larger files. By default, the maximum number will be 5 times the `rps` with a hard ceiling of `2000` but setting this value will override accordingly.
* `batchPerHost` - Sets k6 to allow unlimited requests to the same host in [batch](https://grafana.com/docs/k6/latest/javascript-api/k6-http/batch) calls.

Note that it's best practice to set the number of users (VUs) to the same amount as RPS to ensure the target RPS can be reached.

Options config files typically should be saved to the `k6/config/options` director, although you can save it elsewhere if desired.

#### Selecting which Options file to use

As detailed above the Options file tells the tool how to run the tests. GPT comes with a large selection of out of the box Options files and typically one of these will suit most needs. Selecting the correct options file will depend on the target GitLab environment size and the RPS it would be expected to handle.

Through evaluating real usage data of GitLab along with adding a good amount of headroom that is typical in performance testing, **we've determined that for every 1000 users 20 RPS maximum should be added to the target**. Note however that this target changes depending on the endpoint type being tested. API usage is typically much higher than Web and Git endpoints, so GPT will adjust the RPS automatically depending on the endpoint type as follows:

API: 20 RPS
Web: 2 RPS
Git (Pull): 2 RPS
Git (Push): 0.4 RPS (automatically rounded up to nearest integer)

In addition to the above we've also found that 60 seconds per test is generally enough to get a good result as well as give the tests enough time to handle slower endpoints.

As such, the following options files are recommended to be used based on your target environment user count:

* 1k - `60s_20rps.json`
* 2k - `60s_40rps.json`
* 3k - `60s_60rps.json`
* 5k - `60s_100rps.json`
* 10k - `60s_200rps.json`
* 25k - `60s_500rps.json`
* 50k - `60s_1000rps.json`

The filenames reflect the maximum RPS target GPT is instructed to reach, but again this depends on endpoint type and is adjusted automatically based on the max target as detailed above.

### Tests

Finally, we have the third and last piece of configuration - the [k6 test scripts](https://grafana.com/docs/k6/latest/get-started/running-k6/#execution-modes). Each file contains a test to run against the environment along with any extra config such as setting thresholds.

To get more detailed information about the current test list you can refer to the [Current Test Details wiki page](https://gitlab.com/gitlab-org/quality/performance/wikis/current-test-details).

Like Options, test files are native [k6 test scripts](https://grafana.com/docs/k6/latest/get-started/running-k6/#execution-modes) and all valid k6 features and options can be used here.

With the tool, we provide various curated tests and libraries that are designed to performance test a wide range of GitLab functions. We continue to iterate and release tests also. In each test we set the actions to take (e.g. call an API) along with defining thresholds that determine if the test is a success (e.g. actual RPS should be no less than 20% of the target and no more than 5% of requests made can be failures).

As an example, the following is one of our API Tests, [`api_v4_projects_project.js`](../k6/tests/api/api_v4_projects_project.js), that tests the [GET Single Project API](https://docs.gitlab.com/ee/api/projects.html#get-single-project):

```js
/*global __ENV : true  */
/*
@endpoint: `GET /projects/:id`
@description: [Get single project](https://docs.gitlab.com/ee/api/projects.html#get-single-project)
@gpt_data_version: 1
*/

import http from "k6/http";
import { group } from "k6";
import { Rate } from "k6/metrics";
import { logError, getRpsThresholds, getTtfbThreshold, getLargeProjects, selectRandom } from "../../lib/gpt_k6_modules.js";

export let rpsThresholds = getRpsThresholds()
export let ttfbThreshold = getTtfbThreshold()
export let successRate = new Rate("successful_requests")
export let options = {
  thresholds: {
    "successful_requests": [`rate>${__ENV.SUCCESS_RATE_THRESHOLD}`],
    "http_req_waiting": [`p(90)<${ttfbThreshold}`],
    "http_reqs": [`count>=${rpsThresholds['count']}`]
  }
};

export let projects = getLargeProjects(['encoded_path']);

export function setup() {
  console.log('')
  console.log(`RPS Threshold: ${rpsThresholds['mean']}/s (${rpsThresholds['count']})`)
  console.log(`TTFB P90 Threshold: ${ttfbThreshold}ms`)
  console.log(`Success Rate Threshold: ${parseFloat(__ENV.SUCCESS_RATE_THRESHOLD)*100}%`)
}

export default function() {
  group("API - Project Overview", function() {
    let project = selectRandom(projects);

    let params = { headers: { "Accept": "application/json", "PRIVATE-TOKEN": `${__ENV.ACCESS_TOKEN}` } };
    let res = http.get(`${__ENV.ENVIRONMENT_URL}/api/v4/projects/${project['encoded_path']}`, params);
    /20(0|1)/.test(res.status) ? successRate.add(true) : (successRate.add(false), logError(res));
  });
}
```

The above script is to test the Projects API, namely to [get the details of a specific project](https://docs.gitlab.com/ee/api/projects.html#get-single-project).

The script does the following:

* Informs `eslint` that global environment variables are to be used.
* Sets some tags describing the test that are used by the tool for filtering tests as well as by GitLab Quality for [reporting](https://gitlab.com/gitlab-org/quality/performance/wikis/current-test-details).
* Imports the various k6 libraries and our own custom modules that we use in the script
* Fails the test if the `ACCESS_TOKEN` environment variable hasn't be set on the machine as this test requires it to authenticate. (more details on the token can be found in the [Running Tests](#running-the-tests-with-the-tool) section)
* Configures the Thresholds that will be used to determine if the test has passed or not (more details on the thresholds can be found in the [Test Results](#test-output-and-results) section).
* Loads in the Projects defined in the Environment config file that have the `name` and `group` variables defined. If none found the test is skipped and if multiple is found the test will perform runs against each.
* Logs some useful info in the k6 pre function `setup()` that the tool will use for reporting.
* Finally the main test script itself is last. It selects a Project at random, sets the required headers and then calls the endpoint on the Environment. It then checks if the response was valid, adds the result to the threshold and calls our custom module to report any errors once in the test output.

Note that the above is an example of a typical test. Other [tests](https://gitlab.com/gitlab-org/quality/performance/wikis/current-test-details) may vary in their approach depending on the area being targeted.

#### Test Types

The Tool provides several different types of performance tests that aim to cover the GitLab application. Depending on the type of test you'll see the tool adjust the RPS target accordingly (via adjusting the main RPS target by percentage) as each have a different target throughput based on user patterns.

There are three main types of tests that the Tool provides:

* [`API`](../k6/tests/api) - Tests that target [API](https://docs.gitlab.com/ee/api/) endpoints (RPS target: 100%)
* [`Git`](../k6/tests/git) - Tests that target Git endpoints (RPS target: 10%)
* [`Web`](../k6/tests/web) - Tests that target Web page endpoints (RPS target: 10%)
  * Note that the underlying software this tool is based on, k6, [only tests the response time of the Web page source and doesn't render the page itself](https://docs.k6.io/docs/welcome#section-k6-does-not). This means it won't performance test any HTTP calls found in dynamic scripts, etc... More accurate web page tests could be emulated via [k6's HAR conversion tool](https://grafana.com/docs/k6/latest/using-k6/test-authoring/create-tests-from-recordings/using-the-har-converter) but again note this would only measure response time and not rendering time. To measure Rendering time of pages we recommend [sitespeed.io](https://www.sitespeed.io/).

In addition to the above we have experimental [Scenario](../k6/tests/scenarios) tests that aim to represent a complete user scenario (e.g. creating a new issue). These tests are being skipped by default, see more information in the [Unsafe Tests](#unsafe-tests) section below. Some tests are [Quarantined](../k6/tests/quarantined) due to some ongoing issue with endpoint or test itself. They also don't run unless specifically instructed. There are also [Experimental Tests](../k6/tests/experimental) which contains test for functionalities that are experimental or not GA.

Finally, some specific tests also have individual supporting documentation. These can be found here - [Test Documentation](test_docs/README.md).

#### Unsafe Tests

Some tests provided with the Tool perform [unsafe http requests](https://developer.mozilla.org/en-US/docs/Glossary/safe) (POST, PUT, DELETE or PATCH) to [REST API](https://docs.gitlab.com/ee/api/#rest-api) and will create data on the environment. As such, these tests are not included by default when running the Tool. [Scenario](../k6/tests/scenarios) tests are an example of unsafe tests.

To run these tests the `--unsafe` flag should be used (refer to the Tools help output below).

These tests are typically fine to run against your environment if you're using the recommended `gitlabhq` project or any other project that can be easily removed after testing has completed. It's unrecommended to run these tests against a permanent project or a live GitLab environment.

#### Rate Limit tests

GitLab provides the ability to configure multiple [rate limits](https://docs.gitlab.com/ee/security/rate_limits.html).

Some tests provided with the Tool require specific rate limits to be disabled to get accurate results. By default, GPT will skip
running tests that require rate limit updates unless they have been already disabled on environment. Below is the list of rate limit settings:

```txt
issues_create_limit
raw_blob_request_limit
search_rate_limit
search_rate_limit_unauthenticated
```

GPT has an option to automatically disable rate limits using [Application Setting API](https://docs.gitlab.com/ee/api/settings.html) for tests which require it prior to running the tests and then restoring limits back to original settings. To enable this option, `--rate-limits` flag should be used (refer to the Tools help output below).

Alternatively, listed rate limits can be disabled manually by following respective guidance at [rate limits](https://docs.gitlab.com/ee/security/rate_limits.html).

## Running the Tests with the Tool

When all of the above configuration is in place the tests can be run with the tool. This can be done in one of two ways - With our [Docker image](https://hub.docker.com/r/gitlab/gitlab-performance-tool) (recommended) or natively on a Linux based system.

### Docker (Recommended)

The recommended way to run the Tool is with our Docker image, [gitlab/gitlab-performance-tool](https://hub.docker.com/r/gitlab/gitlab-performance-tool) (alternative mirror: [registry.gitlab.com/gitlab-org/quality/performance/gitlab-performance-tool](https://gitlab.com/gitlab-org/quality/performance/container_registry)), which will be regularly updated with new features and tests. It can also be used in environment that are [offline](https://docs.gitlab.com/ee/user/application_security/offline_deployments/) or running in other network conditions such as behind an HTTP proxy.

The image will start running the tests when it's called. The full options for running the tool can be seen by getting the help output by running `docker run -it gitlab/gitlab-performance-tool --help`:

```txt
GitLab Performance Tool (GPT) v2.15.1 - Performance test runner for GitLab environments based on k6

Documentation: https://gitlab.com/gitlab-org/quality/performance/blob/main/docs/README.md

Usage: run-k6 [options]
Options:
  -e, --environment=<s>     Name of Environment Config file in environments directory that the test(s) will be run with. Alternative filepath can also be given.
  -o, --options=<s>         Name of Options Config file in options directory that the test(s) will be run with. Alternative filepath can also be given. (Default: 20s_2rps.json)
  -t, --tests=<s+>          Names of Test files or directories to run with. When directory given tests will be recursively added from api, web and git subdirs. (Default: tests)
  -s, --scenarios           Include any tests inside the test directory's scenarios subfolder when true.
  -q, --quarantined         Include any tests inside the test directory's quarantined subfolder when true.
  -v, --vulnerabilities     Include any tests that test vulnerability report/api
  -x, --excludes=<s+>       List of words used to exclude tests by matching against their names.
  -u, --unsafe              Include any tests that perform unsafe requests (POST, PUT, DELETE, PATCH)
  -r, --rate-limits         Allow GPT to disable certain rate limits before the tests and restore limits after the test run.
  -p, --experimental        Include any tests inside the test directory's experimental subfolder when true.
  -n, --unattended          Skip all user prompts and run through tests automatically.
  -i, --influxdb-url=<s>    URL of an Influx DB server where GPT can optionally upload test run statistics.
  -h, --help                Show this help message
  --version                 Print version and exit

Environment Variable(s):
  ACCESS_TOKEN                A valid GitLab Personal Access Token for the specified environment. The token should come from a User that has admin access for the project(s) to be tested and have all permissions set. (Default: nil)
  GPT_DEBUG                   Shows debug output when set to true. (Default: nil)
  GPT_SKIP_RETRY              Skip failed test retry when set to true. (Default: nil)
  GPT_TTFB_P95                Add TTFB 95 to the test results output. (Default: nil)
  GPT_RESULTS_SORT_BY_FAILED  Sort GPT results by failed tests first. (Default: nil)

Examples:
  Run all Tests with the 60s_200rps Options file against the 10k Environment:
    docker run -it gitlab/gitlab-performance-tool --environment 10k.json --options 60s_200rps.json
  Run all API Tests with the 60s_200rps Options file against the 10k Environment:
    docker run -it gitlab/gitlab-performance-tool --environment 10k.json --options 60s_200rps.json --tests api
  Run a specific Test with the 60s_200rps Options file against the 10k Environment:
    docker run -it gitlab/gitlab-performance-tool --environment 10k.json --options 60s_200rps.json --tests api_v4_groups_projects.js
```

As standard with Docker you can mount several volumes to get your own config \ test files into the container and results out. The image provides several specific mount points for you to add your own files (in addition to the default ones) as detailed below:

* root (/)
  * `/config` - For any additional config files (each in their own respective subfolder).
    * `/environments` - [Environment Config files](environment_prep.md##preparing-the-environment-file).
    * `/options` - [Options Config files](../k6/config/options)
    * `/projects` - [Project Config files](environment_prep.md#using-custom-large-projects) (only required when you're using a custom Large Project).
  * `/results` - Contains any result files after test runs
  * `/tests` - For any additional tests files.

Here's an example of how you would run the Docker image with all pieces of config and results mounted (replacing placeholders as appropriate):

```sh
docker run -it \
  -e ACCESS_TOKEN=<TOKEN> \
  -v <HOST CONFIG FOLDER>:/config \
  -v <HOST TESTS FOLDER>:/tests \
  -v <HOST RESULTS FOLDER>:/results \
  gitlab/gitlab-performance-tool --environment <ENV FILE NAME>.json --options 60s_500rps.json
```

After running the results will be in the folder you mounted. More details on the results can be found in the [Test Output and Results](#test-output-and-results).

See more command examples in the help output by running `docker run -it gitlab/gitlab-performance-tool --help`.

### Linux \ macOS

You can also run the tool natively on a Linux \ macOS machine with some caveats:

* The tool has been tested on Debian and Alpine based distros for Linux and recent versions of macOS. Other versions should also still work, but your milage may vary.
* This method will require the machine running the tool to have internet access to install Ruby Gems and k6 (if not already present).

Before running some setup is required for the tool:

1. That [Git LFS](https://git-lfs.github.com/) is installed and any LFS data is confirmed pulled via `git lfs pull`.
1. First, set up [`Ruby`](https://www.ruby-lang.org/en/documentation/installation/) using Ruby version specified in [`.tool-versions` file](../.tool-versions) and [`Ruby Bundler`](https://bundler.io) if they aren't already available on the machine.
1. Next, install the required Ruby Gems via Bundler
    * `bundle install`

Next you would need to add any of your custom Environment, Options and Test files to their respective directories. From the tool's root folder this would be `k6/config/environments`, `k6/config/options` and `k6/tests` respectively.

Once setup is done you can run the tool with the `bin/run-k6` script. The options for running the tests are the same as when running in Docker, but the examples change to the following:

```txt
Examples:
  Run all Tests with the 60s_200rps Options file against the 10k Environment:
    ./bin/run-k6 --environment 10k.json --options 60s_200rps.json
  Run all API Tests with the 60s_200rps Options file against the 10k Environment:
    ./bin/run-k6 --environment 10k.json --options 60s_200rps.json --tests api
  Run a specific Test with the 60s_200rps Options file against the 10k Environment:
    ./bin/run-k6 --environment 10k.json --options 60s_200rps.json --tests api_v4_groups_projects.js
```

After running the results will be in the tool's `results` folder. More details on the results can be found in [Test Output and Results](#test-output-and-results).

### Running the specific test

GPT can execute specific tests or test subsets. Use the `--tests` option to specify which tests to run. For example:

* Run a single test file by name (Docker example):

```shell
docker run -it gitlab/gitlab-performance-tool --environment 10k.json --options 60s_200rps.json --tests api_v4_groups_projects.js
```

* Run all tests in a specific [test folder](../k6/tests/) (Linux \ macOS example):

```shell
# runs all tests under 'api' folder
./bin/run-k6 --environment 10k.json --options 60s_200rps.json --tests api
```

* Run tests matching a pattern (Docker example):

```shell
# finds any test containing "projects" in filename
docker run -it gitlab/gitlab-performance-tool --environment 10k.json --options 60s_200rps.json --tests api_v4_groups_projects.js
```

A full list of test can be found on the [Test Types](#test-types) section.

### Test Output and Results

After starting the tool you will see it running each test in order. Once all tests have completed you will be presented with a results summary. As an example, here is a test summary for all tests done against the `10k` environment:

```txt
* Environment:                10k
* Environment Version:        17.4.0-pre `aae05602b3a`
* Option:                     60s_200rps
* Date:                       2024-09-04
* Run Time:                   1h 37m 10.6s (Start: 05:11:18 UTC, End: 06:48:28 UTC)
* GPT Version:                v2.15.0
❯ Overall Results Score: 96.69%
NAME                                                     | RPS   | RPS RESULT           | TTFB AVG  | TTFB P90              | REQ STATUS     | RESULT
---------------------------------------------------------|-------|----------------------|-----------|-----------------------|----------------|--------
api_v4_groups                                            | 200/s | 192.43/s (>128.00/s) | 468.76ms  | 765.55ms (<1700ms)    | 100.00% (>99%) | Passed¹
api_v4_groups_group                                      | 200/s | 24.9/s (>8.00/s)     | 7363.54ms | 15315.78ms (<24000ms) | 100.00% (>20%) | Passed¹
api_v4_groups_group_subgroups                            | 200/s | 176.04/s (>96.00/s)  | 1016.91ms | 1587.16ms (<2000ms)   | 100.00% (>99%) | Passed¹
api_v4_groups_issues                                     | 200/s | 169.41/s (>32.00/s)  | 1042.87ms | 2094.44ms (<8000ms)   | 100.00% (>99%) | Passed¹
api_v4_groups_merge_requests                             | 200/s | 118.85/s (>16.00/s)  | 1537.86ms | 2397.42ms (<17500ms)  | 100.00% (>99%) | Passed¹
api_v4_groups_projects                                   | 200/s | 86.0/s (>8.00/s)     | 2145.22ms | 3505.43ms (<22000ms)  | 100.00% (>30%) | Passed¹
api_v4_projects                                          | 200/s | 46.21/s (>24.00/s)   | 3889.15ms | 5830.78ms (<9000ms)   | 100.00% (>99%) | Passed¹
api_v4_projects_deploy_keys                              | 200/s | 198.76/s (>160.00/s) | 62.89ms   | 87.22ms (<200ms)      | 100.00% (>99%) | Passed
api_v4_projects_issues                                   | 200/s | 195.83/s (>160.00/s) | 254.41ms  | 354.62ms (<1300ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_issues_issue                             | 200/s | 191.17/s (>128.00/s) | 245.55ms  | 277.21ms (<2800ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_issues_search                            | 200/s | 194.12/s (>80.00/s)  | 237.89ms  | 283.25ms (<1900ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_languages                                | 200/s | 198.8/s (>160.00/s)  | 53.20ms   | 60.54ms (<200ms)      | 100.00% (>99%) | Passed
api_v4_projects_merge_requests                           | 200/s | 195.89/s (>160.00/s) | 150.56ms  | 169.28ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_merge_requests_merge_request             | 200/s | 191.72/s (>80.00/s)  | 285.51ms  | 321.14ms (<3500ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_merge_requests_merge_request_commits     | 200/s | 197.38/s (>160.00/s) | 114.01ms  | 131.47ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_merge_requests_merge_request_diffs       | 200/s | 197.68/s (>160.00/s) | 98.83ms   | 112.27ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_merge_requests_merge_request_discussions | 200/s | 195.64/s (>160.00/s) | 215.15ms  | 251.80ms (<1000ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_merge_requests_search                    | 200/s | 188.68/s (>48.00/s)  | 444.28ms  | 663.60ms (<5000ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_project                                  | 200/s | 194.61/s (>160.00/s) | 173.96ms  | 188.74ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_project_pipelines                        | 200/s | 198.15/s (>160.00/s) | 80.20ms   | 88.11ms (<200ms)      | 100.00% (>99%) | Passed
api_v4_projects_project_pipelines_pipeline               | 200/s | 197.69/s (>160.00/s) | 87.84ms   | 100.63ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_project_pipelines_pipeline_jobs          | 200/s | 181.12/s (>32.00/s)  | 981.72ms  | 1465.61ms (<2000ms)   | 100.00% (>99%) | Passed¹
api_v4_projects_project_services                         | 200/s | 198.74/s (>160.00/s) | 71.95ms   | 81.72ms (<200ms)      | 100.00% (>99%) | Passed
api_v4_projects_releases                                 | 200/s | 196.01/s (>160.00/s) | 117.34ms  | 131.07ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_branches                      | 200/s | 195.61/s (>160.00/s) | 79.09ms   | 84.43ms (<200ms)      | 100.00% (>99%) | Passed
api_v4_projects_repository_branches_branch               | 200/s | 197.9/s (>160.00/s)  | 113.11ms  | 125.94ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_branches_search               | 200/s | 197.98/s (>160.00/s) | 76.09ms   | 85.61ms (<200ms)      | 100.00% (>99%) | Passed¹
api_v4_projects_repository_commits                       | 200/s | 198.15/s (>160.00/s) | 107.33ms  | 117.88ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_commits_all                   | 200/s | 183.31/s (>160.00/s) | 139.89ms  | 149.19ms (<200ms)     | 100.00% (>99%) | Passed¹
api_v4_projects_repository_commits_commit                | 200/s | 197.93/s (>160.00/s) | 99.79ms   | 111.02ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_commits_commit_diff           | 200/s | 196.82/s (>128.00/s) | 208.34ms  | 227.30ms (<1500ms)    | 100.00% (>99%) | Passed¹
api_v4_projects_repository_compare                       | 200/s | 195.82/s (>160.00/s) | 108.01ms  | 116.87ms (<200ms)     | 100.00% (>99%) | Passed¹
api_v4_projects_repository_files_file_blame              | 200/s | 39.9/s (>1.60/s)     | 4518.89ms | 6845.76ms (<35000ms)  | 100.00% (>15%) | Passed¹
api_v4_projects_repository_files_file_metadata           | 200/s | 198.22/s (>160.00/s) | 109.68ms  | 120.89ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_files_file_raw                | 200/s | 183.31/s (>160.00/s) | 116.24ms  | 127.03ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_projects_repository_tags                          | 200/s | 198.01/s (>160.00/s) | 116.05ms  | 125.89ms (<200ms)     | 100.00% (>99%) | Passed¹
api_v4_projects_repository_tree                          | 200/s | 197.94/s (>160.00/s) | 119.40ms  | 131.37ms (<200ms)     | 100.00% (>99%) | Passed
api_v4_search_global                                     | 200/s | 192.47/s (>64.00/s)  | 1264.09ms | 2902.44ms (<13000ms)  | 80.42% (>80%)  | Passed¹
api_v4_search_groups                                     | 200/s | 195.2/s (>64.00/s)   | 328.86ms  | 629.55ms (<16000ms)   | 100.00% (>80%) | Passed¹
api_v4_search_projects                                   | 200/s | 194.48/s (>64.00/s)  | 283.59ms  | 512.90ms (<16000ms)   | 100.00% (>80%) | Passed¹
api_v4_summarize_merge_request                           | 200/s | 195.53/s (>160.00/s) | 88.73ms   | 103.31ms (<1000ms)    | 100.00% (>99%) | Passed
git_clone                                                | 8/s   | 1.48/s (>0.26/s)     | 471.86ms  | 1060.42ms (<4000ms)   | 100.00% (>99%) | Passed¹
git_ls_remote                                            | 20/s  | 18.32/s (>16.00/s)   | 73.97ms   | 87.74ms (<200ms)      | 100.00% (>99%) | Passed
git_pull                                                 | 20/s  | 19.49/s (>16.00/s)   | 105.26ms  | 137.14ms (<400ms)     | 100.00% (>99%) | Passed¹
git_push                                                 | 4/s   | 3.44/s (>3.20/s)     | 530.62ms  | 841.20ms (<1000ms)    | 100.00% (>99%) | Passed¹
scenario_api_list_group_variables                        | 1/s   | 0.98/s (>0.64/s)     | 244.53ms  | 446.19ms (<450ms)     | 100.00% (>99%) | Passed¹
scenario_api_list_project_variables                      | 1/s   | 0.98/s (>0.64/s)     | 247.78ms  | 406.41ms (<450ms)     | 100.00% (>99%) | Passed¹
scenario_api_new_branches                                | 1/s   | 0.97/s (>0.64/s)     | 618.36ms  | 810.78ms (<1500ms)    | 100.00% (>99%) | Passed¹
scenario_api_new_commits                                 | 1/s   | 0.99/s (>0.64/s)     | 677.87ms  | 799.97ms (<900ms)     | 100.00% (>99%) | Passed¹
scenario_api_new_group_variables                         | 1/s   | 1.01/s (>0.64/s)     | 215.52ms  | 331.39ms (<450ms)     | 100.00% (>99%) | Passed¹
scenario_api_new_groups                                  | 1/s   | 0.91/s (>0.01/s)     | 794.59ms  | 1297.16ms (<9000ms)   | 100.00% (>99%) | Passed¹
scenario_api_new_issues                                  | 1/s   | 0.84/s (>0.64/s)     | 910.26ms  | 1765.14ms (<2500ms)   | 100.00% (>99%) | Passed
scenario_api_new_project_variables                       | 1/s   | 0.99/s (>0.64/s)     | 244.20ms  | 274.11ms (<650ms)     | 100.00% (>99%) | Passed¹
scenario_api_new_projects                                | 1/s   | 0.52/s (>0.03/s)     | 1867.19ms | 2902.46ms (<7000ms)   | 100.00% (>99%) | Passed¹
scenario_api_update_groups                               | 1/s   | 1.0/s (>0.64/s)      | 302.70ms  | 403.50ms (<900ms)     | 100.00% (>99%) | Passed¹
scenario_api_update_projects                             | 1/s   | 0.95/s (>0.64/s)     | 681.66ms  | 1331.19ms (<1700ms)   | 100.00% (>99%) | Passed¹
web_group                                                | 20/s  | 19.81/s (>16.00/s)   | 198.06ms  | 227.38ms (<500ms)     | 100.00% (>99%) | Passed¹
web_group_issues                                         | 20/s  | 19.7/s (>16.00/s)    | 182.34ms  | 205.56ms (<500ms)     | 100.00% (>99%) | Passed¹
web_group_merge_requests                                 | 20/s  | 19.04/s (>16.00/s)   | 372.99ms  | 500.64ms (<800ms)     | 100.00% (>99%) | Passed¹
web_project                                              | 20/s  | 19.41/s (>16.00/s)   | 298.07ms  | 341.39ms (<800ms)     | 100.00% (>99%) | Passed¹
web_project_branches                                     | 20/s  | 18.84/s (>16.00/s)   | 438.89ms  | 497.52ms (<900ms)     | 100.00% (>99%) | Passed¹
web_project_branches_search                              | 20/s  | 18.58/s (>16.00/s)   | 799.73ms  | 844.94ms (<1600ms)    | 100.00% (>99%) | Passed¹
web_project_commit                                       | 20/s  | 18.84/s (>6.40/s)    | 634.47ms  | 1750.54ms (<3500ms)   | 100.00% (>99%) | Passed¹
web_project_commits                                      | 20/s  | 18.64/s (>16.00/s)   | 526.67ms  | 570.14ms (<1000ms)    | 100.00% (>99%) | Passed¹
web_project_file_blame                                   | 20/s  | 8.48/s (>3.20/s)     | 2079.97ms | 2760.76ms (<3100ms)   | 100.00% (>99%) | Passed¹
web_project_file_rendered                                | 20/s  | 17.76/s (>12.80/s)   | 701.50ms  | 1249.91ms (<1700ms)   | 100.00% (>99%) | Passed¹
web_project_file_source                                  | 20/s  | 19.39/s (>9.60/s)    | 507.96ms  | 623.47ms (<2700ms)    | 100.00% (>99%) | Passed¹
web_project_files                                        | 200/s | 19.77/s (>16.00/s)   | 166.58ms  | 269.61ms (<1200ms)    | 100.00% (>99%) | Passed¹
web_project_issue                                        | 20/s  | 19.5/s (>16.00/s)    | 338.86ms  | 622.17ms (<2000ms)    | 100.00% (>99%) | Passed¹
web_project_issues                                       | 20/s  | 19.53/s (>16.00/s)   | 274.51ms  | 318.12ms (<700ms)     | 100.00% (>99%) | Passed¹
web_project_issues_search                                | 20/s  | 19.5/s (>16.00/s)    | 278.39ms  | 317.24ms (<700ms)     | 100.00% (>99%) | Passed¹
web_project_merge_request                                | 20/s  | 18.0/s (>16.00/s)    | 872.34ms  | 2272.03ms (<2800ms)   | 100.00% (>99%) | Passed¹
web_project_merge_request_changes                        | 20/s  | 18.6/s (>16.00/s)    | 676.18ms  | 911.22ms (<1500ms)    | 100.00% (>99%) | Passed¹
web_project_merge_request_commits                        | 20/s  | 14.59/s (>9.60/s)    | 1199.64ms | 1362.16ms (<1950ms)   | 100.00% (>99%) | Passed¹
web_project_merge_requests                               | 20/s  | 18.51/s (>16.00/s)   | 498.75ms  | 610.93ms (<900ms)     | 100.00% (>99%) | Passed¹
web_project_merge_requests_search                        | 20/s  | 18.94/s (>16.00/s)   | 491.31ms  | 598.88ms (<700ms)     | 100.00% (>99%) | Passed¹
web_project_pipelines                                    | 20/s  | 18.92/s (>12.80/s)   | 335.08ms  | 447.40ms (<800ms)     | 100.00% (>99%) | Passed¹
web_project_pipelines_pipeline                           | 20/s  | 19.19/s (>16.00/s)   | 458.44ms  | 1013.14ms (<2500ms)   | 100.00% (>99%) | Passed¹
web_project_repository_compare                           | 20/s  | 5.26/s (>0.80/s)     | 3451.74ms | 4226.25ms (<7500ms)   | 100.00% (>99%) | Passed¹
web_project_tags                                         | 20/s  | 18.4/s (>12.80/s)    | 774.39ms  | 845.44ms (<1500ms)    | 100.00% (>99%) | Passed¹
web_search_global                                        | 20/s  | 19.83/s (>16.00/s)   | 189.79ms  | 555.40ms (<2000ms)    | 100.00% (>99%) | Passed¹
web_search_groups                                        | 20/s  | 19.9/s (>16.00/s)    | 87.73ms   | 85.82ms (<3000ms)     | 100.00% (>99%) | Passed¹
web_search_projects                                      | 20/s  | 19.53/s (>16.00/s)   | 58.93ms   | 80.20ms (<3000ms)     | 100.00% (>99%) | Passed¹
```

Going through the output step by step:

* First there are stats about the environment, tests and the GPT version.
* Next is the Overall Results Score of the environment. This value is calculated from all of the test results and presented as a distilled value to show how well the environment performed overall. Typically a well performing environment should be above 90%.
* After the score is the main results table for each test run. In this table each column shows the following:
  * `NAME` - The name of the test run. Matches the filename of the test as found in the [`tests`](../k6/tests) folder
  * `RPS` - The RPS target used during the test.
  * `RPS RESULT` - The RPS achieved along with it's passing threshold.
  * `TTFB AVG` - The average [Time To First Byte](https://en.wikipedia.org/wiki/Time_to_first_byte) (TTFB) in ms.
  * `TTFB P90` - The 90th [percentile](https://en.wikipedia.org/wiki/Percentile_rank) of TTFB along with it's passing threshold.
    * `TTFB P95` column could be optionally outputted by passing `GPT_TTFB_P95` environment variable to the GPT.
  * `REQ STATUS` - The percentage of requests made by the test that returned a successful status (HTTP Code 200 / 201 returned) along with it's passing threshold.
  * `RESULT` - The final result of the test based on its thresholds.
* Finally, the output will end with some optional informational notes for the summary depending on how the results went.

Test results are sorted alphabetically by default, grouping tests by endpoints for easier pattern identification and comparison. Results can be optionally sorted by failed tests first by passing `GPT_RESULTS_SORT_BY_FAILED` environment variable to the GPT when running the tests.

Further information on the results output from `k6` can be found over on its documentation - [Results output](https://grafana.com/docs/k6/latest/get-started/results-output).

In addition to standard output in the terminal, results are collected in various formats in the `results/` folder.
Currently supported formats are text, JSON and CSV.

#### Test Thresholds

As described above there are several thresholds that determine if a test has passed or not:

* RPS achieved by the test is above its threshold. The baseline expectation is that actual requests processed should be at least 80% of the target load (with a 20% buffer to account for network or other quirks) by default, but tests can have their own different thresholds set depending on the area they are testing. Refer to each test to confirm thresholds.
* [Time To First Byte](https://en.wikipedia.org/wiki/Time_to_first_byte) (TTFB) response time is below its threshold. The base threshold for endpoints is 200ms, aligning with industry standards, but like RPS this can differ depending on the test.
* Success Rate threshold tracks that more than 99% of all requests made were successful (HTTP Code 200 / 201 returned).

Some endpoints have known performance issues and therefore require custom thresholds. These customized thresholds are tracked with [corresponding performance issues](https://gitlab.com/gitlab-org/quality/performance/-/wikis/current-test-details#known-issues) with different [severity](https://handbook.gitlab.com/handbook/engineering/infrastructure/engineering-productivity/issue-triage/#severity) levels based on the magnitude of the delay. This allows the tests to provide meaningful pass/fail criteria while still monitoring for performance regressions.

#### Evaluating Failures

As part of results output, GPT creates automated `Failure Evaluation Report` Markdown file with the evaluation of individual test failures. This report provides additional insights into why a test failed, such as whether it was due to a small margin of failure, a specific threshold violation or due to another error such as rate limit error. The report can be used as a starting point for failures investigation, but it's not a substitution for a full performance results analysis. Comprehensive analysis requires examining patterns across all tests, system-wide metrics, and full review.

If any of the tests report RPS, TTFB or Status threshold failures these should be evaluated accordingly in line with the following:

* If any of the tests failed but only by a small amount (e.g. within 10% of the threshold) this is likely due to environmental (such as it still starting up) or network conditions such as latency. If further test runs don't report the same errors then these can typically can be ignored.
* If the failures are substantial (e.g. over 50% of the threshold) this would suggest an environment or product issue and further investigation may be required with even possible escalation through the [appropriate channels](https://about.gitlab.com/support/) (e.g. A support ticket or an issue raised against the main GitLab project).
  * Failures in `TTFB P90` and `RPS RESULT` can be investigated further by checking CPU and Memory utilization on target environment and by analysing [GitLab logs](https://docs.gitlab.com/ee/administration/logs/#production_jsonlog) for duration and queueing times for specific test requests.
  * `REQ STATUS` failures can be investigated by checking GPT logs (`_results_output.log`) for response code errors which caused the failures. GPT also outputs Correlation ID if it's available in response so that it can be used for finding the specific error in the GitLab logs.
  * Check the [Current Test Details wiki page](https://gitlab.com/gitlab-org/quality/performance/wikis/current-test-details) for known performance issues and list of GitLab components that are stressed during each test for the reference.
  * If it's a `web` test with no visible failed threshold in Test Result table, check `_results_output.log` file for details which endpoint failed in Web test. For example, `web_project_file_rendered` failed in one of the pipelines - in k6 results we can see that TTFB threshold failed for `✗ { endpoint:file?format=json&viewer=rich }` with `p(90)=1706.65ms` which is over `1700ms` threshold for the test. The GPT framework improvements in outputting multiple endpoint results is tracked at [issue#564](https://gitlab.com/gitlab-org/quality/performance/-/issues/564).
    <details><summary>Click to expand</summary>

      ```sh
      █ Web - Project File Rendered
      data_received.................................: 223 MB  3.7 MB/s
      data_sent.....................................: 1.3 MB  22 kB/s
      group_duration................................: avg=1926.25ms min=748.66ms med=1969.56ms max=3360.26ms p(90)=2749.85ms p(95)=3246.05ms
      http_req_blocked..............................: avg=0.10ms    min=0.00ms   med=0.00ms    max=17.19ms   p(90)=0.00ms    p(95)=0.00ms
      http_req_connecting...........................: avg=0.01ms    min=0.00ms   med=0.00ms    max=0.69ms    p(90)=0.00ms    p(95)=0.00ms
      http_req_duration.............................: avg=519.72ms  min=284.45ms med=485.89ms  max=1864.70ms p(90)=579.01ms  p(95)=1713.07ms
        { expected_response:true }..................: avg=519.72ms  min=284.45ms med=485.89ms  max=1864.70ms p(90)=579.01ms  p(95)=1713.07ms
      http_req_failed...............................: 0.00%   ✓ 0        ✗ 236
      http_req_receiving............................: avg=4.48ms    min=0.12ms   med=2.98ms    max=44.70ms   p(90)=7.13ms    p(95)=8.70ms
      http_req_sending..............................: avg=0.10ms    min=0.04ms   med=0.10ms    max=0.17ms    p(90)=0.12ms    p(95)=0.13ms
      http_req_tls_handshaking......................: avg=0.03ms    min=0.00ms   med=0.00ms    max=1.60ms    p(90)=0.00ms    p(95)=0.00ms
      http_req_waiting..............................: avg=515.15ms  min=283.39ms med=478.70ms  max=1856.55ms p(90)=572.49ms  p(95)=1706.24ms
      ✗ { endpoint:file?format=json&viewer=rich }...: avg=668.05ms  min=457.30ms med=514.60ms  max=1856.55ms p(90)=1706.65ms p(95)=1803.47ms
      ✓ { endpoint:file }...........................: avg=362.25ms  min=283.39ms med=309.61ms  max=1656.95ms p(90)=345.00ms  p(95)=358.63ms
    ✓ http_reqs.....................................: 236     3.903247/s
      ✓ { endpoint:file?format=json&viewer=rich }...: 118     1.951623/s
      ✓ { endpoint:file }...........................: 118     1.951623/s
      iteration_duration............................: avg=1910.10ms min=0.15ms   med=1969.58ms max=3360.30ms p(90)=2747.40ms p(95)=3246.06ms
      iterations....................................: 118     1.951623/s
    ✓ successful_requests...........................: 100.00% ✓ 236      ✗ 0
      vus...........................................: 1       min=1      max=4
      vus_max.......................................: 4       min=4      max=4
      ```

      ```txt
      NAME                                                     | RPS  | RPS RESULT         | TTFB AVG   | TTFB P90              | REQ STATUS     | RESULT

      ---------------------------------------------------------|------|--------------------|------------|-----------------------|----------------|---------
      web_project_file_rendered                                | 4/s  | 3.9/s (>2.56/s)    | 515.15ms   | 572.49ms (<1700ms)    | 100.00% (>99%) | FAILED¹²
      ```

    </details>
* If GPT is being run against an older version of GitLab you may see some failures as thresholds are tailored to the latest version.

In some cases you may see tests reporting as failures in the Results Summary but no obvious reason why due to it being a summary. When evaluating failures it's recommended that the full test output is analyzed as well. There you will likely find the specific reason for the failure, such as one specific endpoint failing if there were multiple ones being tested.

#### Comparing Results

To help further evaluate if results are in line you can compare them to our main Reference Architecture performance test results, which are available publicly in [reference-architectures](https://gitlab.com/gitlab-org/reference-architectures/-/wikis/home) project wiki.

#### Using InfluxDB to store results

GPT provides a custom [InfluxDB writer](https://docs.influxdata.com/influxdb/v1.7/guides/writing_data/#write-data-using-the-influxdb-api) that sends GPT results summary data once the test run is finished:

* Measurements: `ttfb_avg`, `ttfb_p90`, `ttfb_p95`, `rps_result`, `success_rate`, `score`
* Tags: `gitlab_version`, `gitlab_revision`, `gpt_version`, `test_name`
  * Optional `gpt_custom_tag` tag can be assigned to a custom value using `GPT_INFLUXDB_TAG_VALUE` environment variable

To use [built-in k6 InfluxDB](https://grafana.com/docs/k6/latest/results-output/real-time/influxdb) output pass `K6_INFLUXDB_OUTPUT` environment variable along with `--influxdb-url` flag to the GPT.

Alternatively, test results can be stored in [Prometheus](https://prometheus.io/) by using the [Prometheus InfluxDB exporter](https://github.com/prometheus/influxdb_exporter). It collects metrics via an HTTP API, transforms them and exposes data for scraping by Prometheus.

## Troubleshooting

In this section we'll detail any known potential problems when running `k6` and how to manage them.

### `socket: too many open files`

You may see `k6` throw the error `socket: too many open files` many times if you're running a test with particularly high amount of virtual users and throughput, e.g. More than 200 users / RPS.

When this happens it's due to the underlying OS having a limit for how many files can be open at one time, also known as file descriptors. This can be fixed by increasing the number of files that are allowed to be open in the OS. How this is done typically is dependent on the type and version of the OS and you should refer to its documentation. For example though here are links on how to normally do this on [Linux](https://www.tecmint.com/increase-set-open-file-limits-in-linux/) and [macOS](https://medium.com/mindful-technology/too-many-open-files-limit-ulimit-on-mac-os-x-add0f1bfddde) respectively. Another workaround is to run the tests in Docker, which typically have higher limits by default.

### Tests failing due to sign in page redirect

The GPT tests expect all pages to have public visibility, i.e. the target [Projects](https://docs.gitlab.com/ee/user/public_access.html#change-project-visibility) or [User(s)](https://docs.gitlab.com/ee/user/profile/#make-your-user-profile-page-private). If this isn't the case you'll see redirect errors being thrown in the tests:

```sh
time="2022-02-10T12:28:15Z" level=warning msg="Error detected: '<html><body>You are being <a href=\"https://gitlab-test/users/sign_in\">redirected</a>.</body></html>'" source=console
```

When this occurs you should change the target data to be public and rerun the test.

### Rate Limit errors

GitLab provides the ability to configure multiple [rate limits](https://docs.gitlab.com/ee/security/rate_limits.html).

GPT tests may fail with `Rate Limit error caused by '<rate_limit_name>' limit` or `This endpoint has been requested too many times. Try again later.` if rate limits rules on the target GitLab environment affect the tests. To bypass this issue, temporarily disable the specific limit that is affecting the endpoint during the GPT test run. See more information at [Rate Limit tests](#rate-limit-tests)

A list of existing rate limits can be found in [Configurable limits](https://docs.gitlab.com/ee/security/rate_limits.html#configurable-limits) documentation.
